Screen_Main_limits = 11

local socket = require("socket")
local smtp = require("socket.smtp")
local LTN12 = require("ltn12")
local mime = require("mime")
local ssl = require("ssl")

-- Constants - Gmail only
local smtpAddr_Gmail = "smtp.gmail.com"
local smtpPort_Gmail = 465  -- Use SSL port

local smtpAddr = smtpAddr_Gmail
local smtpPort = smtpPort_Gmail

local rcpt = {}

-- Display recipient list information
function showRcptList()
    we_bas_setstring("@W_HDW300", rcpt[1] or "")
    we_bas_setstring("@W_HDW340", rcpt[2] or "")
    we_bas_setstring("@W_HDW380", rcpt[3] or "")
    we_bas_setstring("@W_HDW420", rcpt[4] or "")
    we_bas_setstring("@W_HDW460", rcpt[5] or "")
end

function addRcptList()
    local newCreceiver = we_bas_getstring("@W_HDW500", 32)
    if #newCreceiver == 0 then return end 
    
    table.insert(rcpt, "<" .. newCreceiver .. ">")
    for i, v in pairs(rcpt) do 
        print(i, v)
    end
    showRcptList()
end

-- Safe send function to handle SSL blocking
function safeSend(sslConn, data, operation)
    local result, err, partial
    local retries = 3
    local timeout = 30  -- Increase timeout
    
    for i = 1, retries do
        if operation == "send" then
            result, err, partial = sslConn:send(data)
        else
            result, err, partial = sslConn:receive()
        end
        
        if result then
            return result, nil, partial
        elseif err == "timeout" or err == "wantwrite" or err == "wantread" then
            print(string.format("SSL %s blocked (%s), retry %d...", 
                  operation, err, i))
            socket.sleep(0.5)  -- Wait 0.5 seconds before retry
        else
            return nil, err, partial
        end
    end
    
    return nil, string.format("%s failed, still blocked after %d retries", operation, retries), partial
end

-- Read file and convert to base64 encoding
function readFileAsBase64(filePath)
    print("Reading file: " .. filePath)
    local file, err = io.open(filePath, "rb")
    if not file then
        print("Failed to open file: " .. (err or "Unknown error"))
        return nil, err
    end
    
    local content = file:read("*a")
    file:close()
    
    if not content then
        print("Failed to read file content")
        return nil, "Failed to read file content"
    end
    
    -- Encode file content to base64
    local base64Content = mime.b64(content)
    print("File read successfully, size: " .. #content .. " bytes, base64 length: " .. #base64Content)
    
    return base64Content
end

-- Split base64 string into lines suitable for SMTP transmission
function wrapBase64Content(base64Content, lineLength)
    lineLength = lineLength or 76
    local wrapped = {}
    for i = 1, #base64Content, lineLength do
        table.insert(wrapped, base64Content:sub(i, i + lineLength - 1))
    end
    return wrapped
end

-- Send large data in chunks
function sendChunkedData(sslConn, data, chunkSize)
    chunkSize = chunkSize or 1024  -- Default 1KB chunk size
    local totalSize = #data
    local sent = 0
    
    while sent < totalSize do
        local chunk = data:sub(sent + 1, math.min(sent + chunkSize, totalSize))
        local result, err = safeSend(sslConn, chunk, "send")
        if not result then
            return false, err
        end
        sent = sent + #chunk
        print(string.format("Sent %d/%d bytes (%.1f%%)", sent, totalSize, (sent/totalSize)*100))
        
        -- Small delay to avoid sending too fast
        if sent < totalSize then
            socket.sleep(0.01)
        end
    end
    
    return true
end

function sendEmail()
    print("=== Start sending email ===")
    
    -- Get email information
    local emailHeaders = {
        sender = we_bas_getstring("@W_HDW200", 32),   -- Sender
        to = we_bas_getstring("@W_HDW1000", 128),     -- Recipient
        cc = we_bas_getstring("@W_HDW1200", 128),     -- CC
        subject = we_bas_getstring("@W_HDW1400", 256) -- Subject
    }
    local emailBody = we_bas_getstring("@W_HDW1700", 256) -- Email body
    local username = we_bas_getstring("@W_HDW200", 32)
    local password = we_bas_getstring("@W_HDW240", 32)
    local emailbodyFilePath = we_bas_getstring("@W_HDW2000", 128) -- Attachment file path
    
    -- Print email information
    print("Sender: " .. emailHeaders.sender)
    print("Recipient: " .. emailHeaders.to)
    print("CC: " .. emailHeaders.cc)
    print("Subject: " .. emailHeaders.subject)
    print("Body length: " .. #emailBody)
    print("Username: " .. username)
    print("Password length: " .. #password)
    print("Attachment file path: " .. emailbodyFilePath)
    print("Recipient list: ")
    for i, recipient in ipairs(rcpt) do
        print("  " .. i .. ": " .. recipient)
    end
    
    -- Read attachment file
    local attachmentBase64, attachmentErr
    local hasAttachment = false
    if emailbodyFilePath and #emailbodyFilePath > 0 then
        attachmentBase64, attachmentErr = readFileAsBase64(emailbodyFilePath)
        if attachmentBase64 then
            hasAttachment = true
            print("Attachment loaded successfully")
        else
            print("Failed to load attachment: " .. (attachmentErr or "Unknown error"))
            we_bas_setstring("@W_HDW3000", "Attachment load failed: " .. (attachmentErr or "Unknown error"))
            return
        end
    else
        print("No attachment specified")
    end
    
    -- Try simpler method: create SSL connection directly
    print("1. Create TCP connection...")
    local tcp = socket.tcp()
    if not tcp then
        print("Error: Failed to create TCP connection")
        we_bas_setstring("@W_HDW3000", "Failed to create TCP connection")
        return
    end
    
    -- Set longer timeout
    tcp:settimeout(30)
    
    print("2. Connect to SMTP server: " .. smtpAddr .. ":" .. smtpPort)
    local result, err = tcp:connect(smtpAddr, smtpPort)
    if not result then
        print("Connection failed: " .. (err or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "Connection failed: " .. (err or "Unknown error"))
        return
    end
    print("3. TCP connection established successfully")
    
    print("4. Create SSL context...")
    local sslParams = {
        mode = "client",
        protocol = "tlsv1_2",
        verify = "none",
        options = {"all", "no_sslv2", "no_sslv3", "no_tlsv1"}
    }
    
    print("5. Wrap SSL connection...")
    local sslConn, sslErr = ssl.wrap(tcp, sslParams)
    if not sslConn then
        print("SSL wrapping failed: " .. (sslErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "SSL wrapping failed: " .. (sslErr or "Unknown error"))
        return
    end
    print("6. SSL wrapping successful")
    
    sslConn:settimeout(30)  -- Increase SSL timeout
    
    print("7. Start SSL handshake...")
    local success, handshakeErr = sslConn:dohandshake()
    if not success then
        print("SSL handshake failed: " .. (handshakeErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "SSL handshake failed: " .. (handshakeErr or "Unknown error"))
        return
    end
    print("8. SSL handshake successful")
    
    -- Read server welcome message
    print("9. Read server welcome message...")
    local welcome, welcomeErr = safeSend(sslConn, nil, "receive")
    if not welcome then
        print("Failed to read welcome message: " .. (welcomeErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "Failed to read welcome message: " .. (welcomeErr or "Unknown error"))
        return
    end
    print("Server welcome: " .. welcome)
    
    -- Execute SMTP protocol manually
    print("10. Send EHLO command...")
    local ehloResult, ehloErr = safeSend(sslConn, "EHLO " .. smtpAddr .. "\r\n", "send")
    if not ehloResult then
        print("EHLO send failed: " .. (ehloErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "EHLO send failed: " .. (ehloErr or "Unknown error"))
        return
    end
    
    print("11. Read EHLO response...")
    local ehloResponse, ehloRespErr = safeSend(sslConn, nil, "receive")
    if not ehloResponse then
        print("EHLO response read failed: " .. (ehloRespErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "EHLO response read failed: " .. (ehloRespErr or "Unknown error"))
        return
    end
    print("EHLO response: " .. ehloResponse)
    
    -- Continue reading multi-line EHLO response
    while true do
        local line, lineErr = safeSend(sslConn, nil, "receive")
        if not line then
            print("EHLO multi-line response read failed: " .. (lineErr or "Unknown error"))
            break
        end
        print("EHLO response: " .. line)
        if string.sub(line, 1, 4) == "250 " then
            break
        end
    end
    
    print("12. Start AUTH LOGIN...")
    local authResult, authErr = safeSend(sslConn, "AUTH LOGIN\r\n", "send")
    if not authResult then
        print("AUTH LOGIN send failed: " .. (authErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "AUTH LOGIN send failed: " .. (authErr or "Unknown error"))
        return
    end
    
    print("13. Read AUTH response...")
    local authResponse, authRespErr = safeSend(sslConn, nil, "receive")
    if not authResponse then
        print("AUTH response read failed: " .. (authRespErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "AUTH response read failed: " .. (authRespErr or "Unknown error"))
        return
    end
    print("AUTH response: " .. authResponse)
    
    -- Send username
    print("14. Send username...")
    local userResult, userErr = safeSend(sslConn, mime.b64(username) .. "\r\n", "send")
    if not userResult then
        print("Username send failed: " .. (userErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "Username send failed: " .. (userErr or "Unknown error"))
        return
    end
    
    print("15. Read username response...")
    local userResp, userRespErr = safeSend(sslConn, nil, "receive")
    if not userResp then
        print("Username response read failed: " .. (userRespErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "Username response read failed: " .. (userRespErr or "Unknown error"))
        return
    end
    print("Username response: " .. userResp)
    
    -- Send password
    print("16. Send password...")
    local passResult, passErr = safeSend(sslConn, mime.b64(password) .. "\r\n", "send")
    if not passResult then
        print("Password send failed: " .. (passErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "Password send failed: " .. (passErr or "Unknown error"))
        return
    end
    
    print("17. Read login result...")
    local loginResp, loginRespErr = safeSend(sslConn, nil, "receive")
    if not loginResp then
        print("Login result read failed: " .. (loginRespErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "Login result read failed: " .. (loginRespErr or "Unknown error"))
        return
    end
    print("Login result: " .. loginResp)
    
    -- Check if login was successful
    if not string.match(loginResp, "^235 ") then
        print("Login failed: " .. loginResp)
        we_bas_setstring("@W_HDW3000", "Login failed: " .. loginResp)
        return
    end
    
    print("18. Login successful, start sending email...")
    
    -- Send MAIL FROM command
    print("19. Send MAIL FROM command...")
    local mailFromCmd = "MAIL FROM: <" .. username .. ">\r\n"
    local mailFromResult, mailFromErr = safeSend(sslConn, mailFromCmd, "send")
    if not mailFromResult then
        print("MAIL FROM send failed: " .. (mailFromErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "MAIL FROM send failed: " .. (mailFromErr or "Unknown error"))
        return
    end
    
    print("20. Read MAIL FROM response...")
    local mailFromResp, mailFromRespErr = safeSend(sslConn, nil, "receive")
    if not mailFromResp then
        print("MAIL FROM response read failed: " .. (mailFromRespErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "MAIL FROM response read failed: " .. (mailFromRespErr or "Unknown error"))
        return
    end
    print("MAIL FROM response: " .. mailFromResp)
    
    -- Check MAIL FROM response
    if not string.match(mailFromResp, "^250 ") then
        print("MAIL FROM failed: " .. mailFromResp)
        we_bas_setstring("@W_HDW3000", "MAIL FROM failed: " .. mailFromResp)
        return
    end
    
    -- Send RCPT TO command (for each recipient)
    print("21. Send RCPT TO command...")
    for i, recipient in ipairs(rcpt) do
        print("  Send RCPT TO: " .. recipient)
        local rcptToCmd = "RCPT TO: " .. recipient .. "\r\n"
        local rcptToResult, rcptToErr = safeSend(sslConn, rcptToCmd, "send")
        if not rcptToResult then
            print("RCPT TO send failed: " .. (rcptToErr or "Unknown error"))
            we_bas_setstring("@W_HDW3000", "RCPT TO send failed: " .. (rcptToErr or "Unknown error"))
            return
        end
        
        print("  Read RCPT TO response...")
        local rcptToResp, rcptToRespErr = safeSend(sslConn, nil, "receive")
        if not rcptToResp then
            print("RCPT TO response read failed: " .. (rcptToRespErr or "Unknown error"))
            we_bas_setstring("@W_HDW3000", "RCPT TO response read failed: " .. (rcptToRespErr or "Unknown error"))
            return
        end
        print("  RCPT TO response: " .. rcptToResp)
        
        -- Check RCPT TO response
        if not string.match(rcptToResp, "^250 ") then
            print("RCPT TO failed: " .. rcptToResp)
            we_bas_setstring("@W_HDW3000", "RCPT TO failed: " .. rcptToResp)
            return
        end
    end
    
    -- Send DATA command
    print("22. Send DATA command...")
    local dataResult, dataErr = safeSend(sslConn, "DATA\r\n", "send")
    if not dataResult then
        print("DATA send failed: " .. (dataErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "DATA send failed: " .. (dataErr or "Unknown error"))
        return
    end
    
    print("23. Read DATA response...")
    local dataResp, dataRespErr = safeSend(sslConn, nil, "receive")
    if not dataResp then
        print("DATA response read failed: " .. (dataRespErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "DATA response read failed: " .. (dataRespErr or "Unknown error"))
        return
    end
    print("DATA response: " .. dataResp)
    
    -- Check DATA response
    if not string.match(dataResp, "^354 ") then
        print("DATA failed: " .. dataResp)
        we_bas_setstring("@W_HDW3000", "DATA failed: " .. dataResp)
        return
    end
    
    -- Build email content - use unified multipart format
    print("24. Build email content...")
    local emailContent = {}
    
    -- Generate MIME boundary
    local boundary = "----=_NextPart_" .. tostring(math.random(100000, 999999)) .. "_" .. tostring(socket.gettime()):gsub("%.", "")
    
    -- Add email headers
    table.insert(emailContent, "From: " .. emailHeaders.sender .. "\r\n")
    table.insert(emailContent, "To: " .. emailHeaders.to .. "\r\n")
    if emailHeaders.cc and #emailHeaders.cc > 0 then
        table.insert(emailContent, "Cc: " .. emailHeaders.cc .. "\r\n")
    end
    table.insert(emailContent, "Subject: " .. emailHeaders.subject .. "\r\n")
    table.insert(emailContent, "Date: " .. os.date("!%a, %d %b %Y %H:%M:%S +0000") .. "\r\n")
    table.insert(emailContent, "MIME-Version: 1.0\r\n")
    
    if hasAttachment then
        -- Multipart email (contains text and attachment)
        table.insert(emailContent, "Content-Type: multipart/mixed; boundary=\"" .. boundary .. "\"\r\n")
        table.insert(emailContent, "\r\n")  -- Empty line separates headers and body
        
        -- Text part
        table.insert(emailContent, "--" .. boundary .. "\r\n")
        table.insert(emailContent, "Content-Type: text/plain; charset=UTF-8\r\n")
        table.insert(emailContent, "Content-Transfer-Encoding: 8bit\r\n")
        table.insert(emailContent, "\r\n")
        table.insert(emailContent, emailBody .. "\r\n")
        table.insert(emailContent, "\r\n")
        
        -- Attachment part
        table.insert(emailContent, "--" .. boundary .. "\r\n")
        table.insert(emailContent, "Content-Type: image/png; name=\"image.png\"\r\n")
        table.insert(emailContent, "Content-Disposition: attachment; filename=\"image.png\"\r\n")
        table.insert(emailContent, "Content-Transfer-Encoding: base64\r\n")
        table.insert(emailContent, "\r\n")
        
        -- Add base64 encoded attachment content
        local wrappedAttachment = wrapBase64Content(attachmentBase64)
        for _, line in ipairs(wrappedAttachment) do
            table.insert(emailContent, line .. "\r\n")
        end
        table.insert(emailContent, "\r\n")
        
        -- End boundary
        table.insert(emailContent, "--" .. boundary .. "--\r\n")
    else
        -- Plain text email (still use multipart format but only contains text part)
        table.insert(emailContent, "Content-Type: multipart/alternative; boundary=\"" .. boundary .. "\"\r\n")
        table.insert(emailContent, "\r\n")  -- Empty line separates headers and body
        
        -- Text part
        table.insert(emailContent, "--" .. boundary .. "\r\n")
        table.insert(emailContent, "Content-Type: text/plain; charset=UTF-8\r\n")
        table.insert(emailContent, "Content-Transfer-Encoding: 8bit\r\n")
        table.insert(emailContent, "\r\n")
        table.insert(emailContent, emailBody .. "\r\n")
        table.insert(emailContent, "\r\n")
        
        -- End boundary
        table.insert(emailContent, "--" .. boundary .. "--\r\n")
    end
    
    -- Add end marker
    table.insert(emailContent, "\r\n.\r\n")
    
    -- Send email content
    print("25. Send email content...")
    local contentToSend = table.concat(emailContent)
    print("Email content size: " .. #contentToSend .. " bytes")
    
    -- Use chunked send function
    local contentResult, contentErr = sendChunkedData(sslConn, contentToSend, 512)  -- Use smaller chunk size
    
    if not contentResult then
        print("Email content send failed: " .. (contentErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "Email content send failed: " .. (contentErr or "Unknown error"))
        return
    end
    
    print("26. Read email send result...")
    local sendResult, sendResultErr = safeSend(sslConn, nil, "receive")
    if not sendResult then
        print("Email send result read failed: " .. (sendResultErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "Email send result read failed: " .. (sendResultErr or "Unknown error"))
        return
    end
    print("Email send result: " .. sendResult)
    
    -- Check email send result
    if not string.match(sendResult, "^250 ") then
        print("Email send failed: " .. sendResult)
        we_bas_setstring("@W_HDW3000", "Email send failed: " .. sendResult)
        return
    end
    
    -- Send QUIT command
    print("27. Send QUIT command...")
    local quitResult, quitErr = safeSend(sslConn, "QUIT\r\n", "send")
    if not quitResult then
        print("QUIT send failed: " .. (quitErr or "Unknown error"))
        we_bas_setstring("@W_HDW3000", "QUIT send failed: " .. (quitErr or "Unknown error"))
        return
    end
    
    print("28. Read QUIT response...")
    local quitResp, quitRespErr = safeSend(sslConn, nil, "receive")
    if quitResp then
        print("QUIT response: " .. quitResp)
    end
    
    -- Close connection
    sslConn:close()
    
    print("=== Email sent successfully ===")
    local successMsg = "Email sent successfully!"
    if hasAttachment then
        successMsg = successMsg .. " (with attachment)"
    end
    we_bas_setstring("@W_HDW3000", successMsg)
end